require('dotenv').config();
const express = require('express');
const http = require('http');
const socketIo = require('socket.io');
const cors = require('cors');
const path = require('path');
const fs = require('fs');
const { v4: uuidv4 } = require('uuid');
const multer = require('multer');
const { OpenAI } = require('openai');
const routes = require('./routes');

// Configuração do Express
const app = express();
app.use(cors());
app.use(express.json());
app.use(express.static(path.join(__dirname, '../client')));

// Configuração do servidor HTTP e Socket.IO
const server = http.createServer(app);
const io = socketIo(server, {
  cors: {
    origin: "*",
    methods: ["GET", "POST"]
  }
});

// Configuração do OpenAI
const openai = new OpenAI({
  apiKey: process.env.OPENAI_API_KEY,
});

// Configuração do Multer para upload de arquivos
const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    const uploadDir = path.join(__dirname, 'uploads');
    if (!fs.existsSync(uploadDir)) {
      fs.mkdirSync(uploadDir);
    }
    cb(null, uploadDir);
  },
  filename: (req, file, cb) => {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1e9);
    cb(null, uniqueSuffix + '-' + file.originalname);
  },
});
const upload = multer({ storage });

// Armazenamento de sessões de transcrição
const sessions = new Map();

// Configuração do Socket.IO
io.on('connection', (socket) => {
  console.log('Cliente conectado:', socket.id);
  
  // Criar nova sessão de transcrição
  socket.on('start-session', () => {
    const sessionId = uuidv4();
    sessions.set(sessionId, {
      transcription: '',
      organizedText: '',
      lastChunkTime: Date.now()
    });
    socket.emit('session-started', { sessionId });
    console.log('Nova sessão criada:', sessionId);
  });
  
  // Receber chunks de áudio
  socket.on('audio-chunk', async (data) => {
    try {
      const { sessionId, audioChunk } = data;
      
      if (!sessions.has(sessionId)) {
        socket.emit('error', { message: 'Sessão não encontrada' });
        return;
      }
      
      const session = sessions.get(sessionId);
      
      // Salvar o chunk de áudio temporariamente
      const chunkPath = path.join(__dirname, 'uploads', `${sessionId}-${Date.now()}.webm`);
      fs.writeFileSync(chunkPath, Buffer.from(audioChunk));
      
      // Processar o chunk com Whisper
      const transcription = await processAudioChunk(chunkPath, session.transcription);
      
      // Atualizar a transcrição da sessão
      session.transcription += ' ' + transcription.text;
      session.lastChunkTime = Date.now();
      
      // Verificar se é hora de organizar o texto (a cada 15 segundos)
      let organizedText = session.organizedText;
      if (Date.now() - session.lastOrganizeTime > 15000 || !session.lastOrganizeTime) {
        organizedText = await organizeText(session.transcription);
        session.organizedText = organizedText;
        session.lastOrganizeTime = Date.now();
      }
      
      // Enviar a transcrição atualizada para o cliente
      socket.emit('transcription-update', {
        transcription: session.transcription,
        organizedText: organizedText
      });
      
      // Limpar o arquivo temporário
      fs.unlinkSync(chunkPath);
      
    } catch (error) {
      console.error('Erro ao processar chunk de áudio:', error);
      socket.emit('error', { message: 'Erro ao processar áudio' });
    }
  });
  
  // Finalizar sessão
  socket.on('end-session', ({ sessionId }) => {
    if (sessions.has(sessionId)) {
      sessions.delete(sessionId);
      console.log('Sessão finalizada:', sessionId);
    }
  });
  
  // Desconexão
  socket.on('disconnect', () => {
    console.log('Cliente desconectado:', socket.id);
  });
});

// Função para processar chunk de áudio com Whisper
async function processAudioChunk(audioPath, previousText) {
  try {
    const transcription = await openai.audio.transcriptions.create({
      file: fs.createReadStream(audioPath),
      model: 'whisper-1',
      prompt: previousText.slice(-200), // Usar os últimos 200 caracteres como contexto
      language: 'pt'
    });
    
    return transcription;
  } catch (error) {
    console.error('Erro na transcrição:', error);
    throw error;
  }
}

// Função para organizar o texto em formato de anamnese
async function organizeText(text) {
  try {
    const prompt = `
Organize o seguinte texto de anamnese em tempo real. Mantenha a estrutura, mas não conclua seções que ainda não foram mencionadas:

"${text}"

Organize em:
1. Queixa Principal
2. História da Doença Atual
3. História Familiar
4. História Pregressa Patológica
5. Medicações de uso contínuo
6. Exame Físico
7. Diagnóstico
8. Conduta
`;
    
    const completion = await openai.chat.completions.create({
      model: 'o3-mini',
      messages: [{ role: 'user', content: prompt }],
    });
    
    return completion.choices[0].message.content;
  } catch (error) {
    console.error('Erro ao organizar texto:', error);
    return text; // Retorna o texto original em caso de erro
  }
}

// Rotas da API
app.use('/', routes);

// Iniciar o servidor
const PORT = process.env.PORT || 3000;
server.listen(PORT, () => {
  console.log(`Servidor rodando na porta ${PORT}`);
});
